<?php
/* --------------------------------------------------------------
   Withdrawal.php 2020-04-21
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\Models;

use DateTime;
use Exception;
use Gambio\Admin\Withdrawal\Interfaces\Withdrawal as WithdrawalInterface;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalOrderId;
use InvalidArgumentException;

/**
 * Class Withdrawal
 *
 * @package Gambio\Admin\Withdrawal\Models
 */
class Withdrawal implements WithdrawalInterface
{
    /**
     * @var WithdrawalId
     */
    private $id;
    
    /**
     * @var OrderDetails
     */
    private $order;
    
    /**
     * @var CustomerDetails
     */
    private $customer;
    
    /**
     * @var DateTime
     */
    private $date;
    
    /**
     * @var string
     */
    private $content;
    
    /**
     * @var bool
     */
    private $createdByAdmin;
    
    /**
     * @var DateTime|null
     */
    private $createdOn;
    
    
    /**
     * Withdrawal constructor.
     *
     * @param WithdrawalId    $id
     * @param OrderDetails    $order
     * @param CustomerDetails $customer
     * @param DateTime        $date
     * @param string          $content
     * @param bool            $createdByAdmin
     * @param DateTime|null   $createdOn
     */
    private function __construct(
        WithdrawalId $id,
        OrderDetails $order,
        CustomerDetails $customer,
        DateTime $date,
        string $content,
        bool $createdByAdmin,
        ?DateTime $createdOn = null
    ) {
        $this->id             = $id;
        $this->order          = $order;
        $this->customer       = $customer;
        $this->date           = $date;
        $this->content        = $content;
        $this->createdByAdmin = $createdByAdmin;
        $this->createdOn      = $createdOn;
    }
    
    
    /**
     * Creates a new Withdrawal instance.
     *
     * @param WithdrawalId    $id
     * @param OrderDetails    $order
     * @param CustomerDetails $customer
     * @param DateTime        $date
     * @param string          $content
     * @param bool            $createdByAdmin
     * @param DateTime|null   $createdOn
     *
     * @return Withdrawal
     *
     * @throws InvalidArgumentException
     */
    public static function create(
        WithdrawalId $id,
        OrderDetails $order,
        CustomerDetails $customer,
        DateTime $date,
        string $content,
        bool $createdByAdmin,
        ?DateTime $createdOn = null
    ): Withdrawal {
        return new self($id, $order, $customer, $date, $content, $createdByAdmin, $createdOn);
    }
    
    
    /**
     * @inheritDoc
     */
    public function id(): ?int
    {
        return $this->id->value();
    }
    
    
    /**
     * @inheritDoc
     */
    public function orderId(): ?int
    {
        return $this->order->id();
    }
    
    
    /**
     * @inheritDoc
     */
    public function orderCreationDate(): ?string
    {
        return $this->order->creationDate();
    }
    
    
    /**
     * @inheritDoc
     */
    public function orderDeliveryDate(): ?string
    {
        return $this->order->deliveryDate();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerId(): ?int
    {
        return $this->customer->id();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerGender(): string
    {
        return $this->customer->gender();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerFirstName(): string
    {
        return $this->customer->firstName();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerLastName(): string
    {
        return $this->customer->lastName();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerStreet(): string
    {
        return $this->customer->street();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerPostcode(): string
    {
        return $this->customer->postcode();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerCity(): string
    {
        return $this->customer->city();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerCountry(): string
    {
        return $this->customer->country();
    }
    
    
    /**
     * @inheritDoc
     */
    public function customerEmail(): string
    {
        return $this->customer->email();
    }
    
    
    /**
     * @inheritDoc
     */
    public function date(string $format = 'Y-m-d H:i:s'): string
    {
        return $this->date->format($format);
    }
    
    
    /**
     * @inheritDoc
     */
    public function content(): string
    {
        return $this->content;
    }
    
    
    /**
     * @inheritDoc
     */
    public function wasCreatedByAdmin(): bool
    {
        return $this->createdByAdmin;
    }
    
    
    /**
     * @inheritDoc
     */
    public function createdOn(string $format = 'Y-m-d H:i:s'): ?string
    {
        return ($this->createdOn !== null) ? $this->createdOn->format($format) : null;
    }
    
    
    /**
     * @inheritDoc
     */
    public function changeOrder(WithdrawalOrderId $orderId): void
    {
        $this->order->changeId($orderId);
    }
    
    
    /**
     * Array serialization.
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'id'             => $this->id(),
            'order'          => $this->order->toArray(),
            'customer'       => $this->customer->toArray(),
            'date'           => $this->date(),
            'content'        => $this->content,
            'createdByAdmin' => $this->createdByAdmin,
            'createdOn'      => $this->createdOn(),
        ];
    }
    
    
    /**
     * Creates a new Withdrawal instance from the provided data.
     *
     * @param array $data
     *
     * @return Withdrawal
     */
    public static function createFromArray(array $data): Withdrawal
    {
        try {
            $date = new DateTime($data['date']);
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for withdrawal date provided.');
        }
        try {
            $createdOn = (isset($data['createdOn'])) ? new DateTime($data['createdOn']) : null;
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for withdrawal creation date provided.');
        }
        
        return Withdrawal::create(WithdrawalId::create($data['id'] ?? null),
                                  OrderDetails::createFromArray($data['order']),
                                  CustomerDetails::createFromArray($data['customer']),
                                  $date,
                                  $data['content'],
                                  $data['createdByAdmin'],
                                  $createdOn);
    }
    
    
    /**
     * @inheritDoc
     */
    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
}